<?php
use App\Database;

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../vendor/autoload.php';

// Autenticación básica manual
$env = parse_ini_file(__DIR__ . '/../.env');
$validUser = $env['API_USER'] ?? '';
$validPass = $env['API_PASS'] ?? '';

if (!isset($_SERVER['PHP_AUTH_USER'])) {
    if (isset($_SERVER['HTTP_AUTHORIZATION']) && stripos($_SERVER['HTTP_AUTHORIZATION'], 'basic') === 0) {
        $encoded = substr($_SERVER['HTTP_AUTHORIZATION'], 6);
        $decoded = base64_decode($encoded);
        if ($decoded) {
            list($_SERVER['PHP_AUTH_USER'], $_SERVER['PHP_AUTH_PW']) = explode(':', $decoded, 2);
        }
    }
}

$inputUser = $_SERVER['PHP_AUTH_USER'] ?? null;
$inputPass = $_SERVER['PHP_AUTH_PW'] ?? null;

if ($inputUser !== $validUser || $inputPass !== $validPass) {
    http_response_code(401);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 401,
        'message' => 'No autorizado, faltan credenciales',
        'data' => []
    ]);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 405,
        'message' => 'Método no permitido',
        'data' => []
    ]);
    exit;
}

// Extraer parámetros de la URL
$uriParts = explode('/', trim(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), '/'));
$index = array_search('get-invoice', $uriParts);

$noOrder = $uriParts[$index + 1] ?? null;
$customerCode = $uriParts[$index + 2] ?? null;

if (!$noOrder || !$customerCode) {
    http_response_code(400);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 400,
        'message' => 'Faltan parámetros en la URL',
        'data' => []
    ]);
    exit;
}

try {
    $conn = Database::connect();

    // Validar existencia de la orden
    $stmt = $conn->prepare("SELECT ESTATUS, CLIENTE, id FROM ot WHERE CONCAT(numero, '-', ANIO) = ?");
    $stmt->execute([$noOrder]);
    $orden = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$orden) {
        http_response_code(404);
        echo json_encode([
            'isSuccess' => false,
            'httpStatusCode' => 404,
            'message' => 'Orden no encontrada',
            'data' => []
        ]);
        exit;
    }

    if ($orden['CLIENTE'] != $customerCode) {
        http_response_code(403);
        echo json_encode([
            'isSuccess' => false,
            'httpStatusCode' => 403,
            'message' => 'La orden existe pero no corresponde al cliente indicado',
            'data' => []
        ]);
        exit;
    }

    $estatus = trim($orden['ESTATUS']);
    $idOt = $orden['id'];
    $isFacturada = ($estatus === 'F');

    if ($isFacturada) {
        // Consulta de datos de la factura
        $sql = "
            SELECT 
                ot.noCoordination AS noCoordination,
                cg.fel_documento AS noInvoice,
                cg.fecha_docto AS dateInvoice,
                cg.vlr_total AS amountInvoce,
                (CASE WHEN cg.vlr_total > cg.vlr_pgdo_total THEN 'Parcialmente pagada' ELSE 'Pagada' END) AS statusInvoce
            FROM ot 
            INNER JOIN tbl_ordfac_h oh ON ot.id = oh.ot
            INNER JOIN cgo_cxc cg ON cg.numero_docto = oh.factura AND cg.tipo_cgo_cxc = oh.tipcgcxc
            WHERE ot.id = ?
        ";
        $stmt = $conn->prepare($sql);
        $stmt->execute([$idOt]);

        $factura = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($factura) {
            // Formatear la fecha al estilo ISO 8601 con milisegundos y Z
            $fechaRaw = $factura['dateInvoice'];
            $fechaObj = new DateTime($fechaRaw);
            $factura['dateInvoice'] = $fechaObj->format('Y-m-d\TH:i:s.000\Z');

            // Limpiar datos vacíos
            $data = array_map(fn($val) => trim($val) !== '' ? trim($val) : ' ', $factura);

            http_response_code(200);
            echo json_encode([
                'isSuccess' => true,
                'httpStatusCode' => 200,
                'message' => 'Consulta exitosa',
                'data' => $data
            ]);
            exit;
        } else {
            http_response_code(404);
            echo json_encode([
                'isSuccess' => false,
                'httpStatusCode' => 404,
                'message' => 'No se encontró información de factura para esta orden',
                'data' => []
            ]);
            exit;
        }

    } else {
        // Orden no facturada
        http_response_code(200);
        echo json_encode([
            'isSuccess' => true,
            'httpStatusCode' => 200,
            'message' => 'Orden no facturada',
            'data' => [
                'noCoordination' => trim($noOrder),
                'noInvoice' => ' ',
                'dateInvoice' => ' ',
                'amountInvoce' => ' ',
                'statusInvoce' => 'No facturada'
            ]
        ]);
        exit;
    }

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 500,
        'message' => 'Error al consultar la información de la factura',
        'data' => [],
        'detalle' => $e->getMessage()
    ]);
    exit;
}
