<?php
use App\Database;

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../vendor/autoload.php';

// Autenticación básica manual
$env = parse_ini_file(__DIR__ . '/../.env');
$validUser = $env['API_USER'] ?? '';
$validPass = $env['API_PASS'] ?? '';

if (!isset($_SERVER['PHP_AUTH_USER'])) {
    if (isset($_SERVER['HTTP_AUTHORIZATION']) && stripos($_SERVER['HTTP_AUTHORIZATION'], 'basic') === 0) {
        $encoded = substr($_SERVER['HTTP_AUTHORIZATION'], 6);
        $decoded = base64_decode($encoded);
        if ($decoded) {
            list($_SERVER['PHP_AUTH_USER'], $_SERVER['PHP_AUTH_PW']) = explode(':', $decoded, 2);
        }
    }
}

$inputUser = $_SERVER['PHP_AUTH_USER'] ?? null;
$inputPass = $_SERVER['PHP_AUTH_PW'] ?? null;

if ($inputUser !== $validUser || $inputPass !== $validPass) {
    http_response_code(401);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 401,
        'message' => 'No autorizado, faltan credenciales',
        'data' => []
    ]);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 405,
        'message' => 'Método no permitido',
        'data' => []
    ]);
    exit;
}

$uriParts = explode('/', trim(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH), '/'));
$index = array_search('get-order-status', $uriParts);

$noOrder = $uriParts[$index + 1] ?? null;
$customerCode = $uriParts[$index + 2] ?? null;

if (!$noOrder || !$customerCode) {
    http_response_code(400);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 400,
        'message' => 'Faltan parámetros en la URL',
        'data' => []
    ]);
    exit;
}

try {
    $conn = Database::connect();

    // Primero verificar si existe la orden sin importar cliente
    $sqlOrderExist = "SELECT ESTATUS, CLIENTE FROM ot WHERE CONCAT(numero, '-', ANIO) = ?";
    $stmtExist = $conn->prepare($sqlOrderExist);
    $stmtExist->execute([$noOrder]);
    $order = $stmtExist->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        http_response_code(404);
        echo json_encode([
            'isSuccess' => false,
            'httpStatusCode' => 404,
            'message' => 'Orden no existe',
            'data' => []
        ]);
        exit;
    }

    // Orden existe, verificar cliente
    if ($order['CLIENTE'] != $customerCode) {
        http_response_code(403);
        echo json_encode([
            'isSuccess' => false,
            'httpStatusCode' => 403,
            'message' => 'La orden existe pero no corresponde al cliente indicado',
            'data' => []
        ]);
        exit;
    }

    // Cliente coincide, devolver estado
    $format = fn($val) => trim($val) !== '' ? trim($val) : 'Sin registro';

    $statusMap = [
        'N' => 'Anulada',
        'A' => 'Abierta',
        'F' => 'Facturada',
        'C' => 'Cerrada',
    ];

    $estatus = $format($order['ESTATUS']);
    $statusOrder = $statusMap[$estatus] ?? 'Estado desconocido';

    http_response_code(200);
    echo json_encode([
        'isSuccess' => true,
        'httpStatusCode' => 200,
        'message' => 'Consulta exitosa',
        'data' => ['statusOrder' => $statusOrder]
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode([
        'isSuccess' => false,
        'httpStatusCode' => 500,
        'message' => 'Error al consultar el estado de la orden',
        'data' => []
    ]);
    exit;
}
